/*
 * Decompiled with CFR 0.152.
 */
package de.tum.in.jbdd;

import de.tum.in.jbdd.BddConfiguration;
import de.tum.in.jbdd.BddImpl;
import de.tum.in.jbdd.BitUtil;
import de.tum.in.jbdd.MathUtil;
import java.util.Arrays;
import java.util.Collection;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;

final class BddCache {
    private static final int BINARY_CACHE_OPERATION_ID_OFFSET = 61;
    private static final int BINARY_OPERATION_AND = 0;
    private static final int BINARY_OPERATION_EQUIVALENCE = 4;
    private static final int BINARY_OPERATION_EXISTS = 6;
    private static final int BINARY_OPERATION_IMPLIES = 3;
    private static final int BINARY_OPERATION_N_AND = 1;
    private static final int BINARY_OPERATION_OR = 2;
    private static final int BINARY_OPERATION_XOR = 5;
    private static final int CACHE_VALUE_BIT_SIZE = 25;
    private static final long CACHE_VALUE_MASK = BitUtil.maskLength(25);
    private static final int TERNARY_CACHE_OPERATION_ID_OFFSET = 63;
    private static final int TERNARY_OPERATION_ITE = 0;
    private static final int NEGATION_CACHE_KEY_OFFSET = 32;
    static final Logger logger = Logger.getLogger(BddCache.class.getName());
    static final Collection<BddCache> cacheShutdownHook = new ConcurrentLinkedDeque<BddCache>();
    private final BddImpl associatedBdd;
    private final CacheAccessStatistics binaryAccessStatistics = new CacheAccessStatistics();
    private final int binaryBinsPerHash;
    private final CacheAccessStatistics composeAccessStatistics = new CacheAccessStatistics();
    private final int composeBinsPerHash;
    private final CacheAccessStatistics satisfactionAccessStatistics = new CacheAccessStatistics();
    private final int satisfactionBinsPerHash;
    private final CacheAccessStatistics ternaryAccessStatistics = new CacheAccessStatistics();
    private final int ternaryBinsPerHash;
    private final CacheAccessStatistics negationAccessStatistics = new CacheAccessStatistics();
    private final int negationBinsPerHash;
    private final CacheAccessStatistics volatileAccessStatistics = new CacheAccessStatistics();
    private final int volatileBinsPerHash;
    private long[] binaryKeyStorage;
    private int[] binaryResultStorage;
    @Nullable
    private int[] composeStorage;
    private int lookupHash = -1;
    private int lookupResult = -1;
    private int[] satisfactionKeyStorage;
    private double[] satisfactionResultStorage;
    private long[] ternaryStorage;
    private long[] negationStorage;
    private int[] volatileKeyStorage;
    private int[] volatileResultStorage;

    BddCache(BddImpl associatedBdd) {
        this.associatedBdd = associatedBdd;
        BddConfiguration configuration = associatedBdd.getConfiguration();
        this.negationBinsPerHash = configuration.cacheNegationBinsPerHash();
        this.binaryBinsPerHash = configuration.cacheBinaryBinsPerHash();
        this.ternaryBinsPerHash = configuration.cacheTernaryBinsPerHash();
        this.satisfactionBinsPerHash = configuration.cacheSatisfactionBinsPerHash();
        this.composeBinsPerHash = configuration.cacheComposeBinsPerHash();
        this.volatileBinsPerHash = configuration.cacheVolatileBinsPerHash();
        this.reallocateNegation();
        this.reallocateBinary();
        this.reallocateTernary();
        this.reallocateSatisfaction();
        this.reallocateCompose();
        this.reallocateVolatile();
        if (configuration.logStatisticsOnShutdown()) {
            logger.log(Level.FINER, "Adding {0} to shutdown hook", this);
            BddCache.addToShutdownHook(this);
        }
    }

    private static void addToShutdownHook(BddCache cache) {
        ShutdownHookLazyHolder.init();
        cacheShutdownHook.add(cache);
    }

    private static long buildBinaryKeyStore(long operationId, long inputNode1, long inputNode2) {
        assert (BitUtil.fits(inputNode1, 25) && BitUtil.fits(inputNode2, 25));
        long store = inputNode1;
        store |= inputNode2 << 25;
        return store |= operationId << 61;
    }

    private static long buildTernaryFirstStore(long operationId, long inputNode1, long inputNode2) {
        assert (BitUtil.fits(inputNode1, 25) && BitUtil.fits(inputNode2, 25));
        return inputNode1 | inputNode2 << 25 | operationId << 63;
    }

    private static long buildTernarySecondStore(long inputNode3, long resultNode) {
        assert (BitUtil.fits(inputNode3, 25) && BitUtil.fits(resultNode, 25));
        return resultNode | inputNode3 << 25;
    }

    private static long buildNegationFullKey(long inputNode) {
        return inputNode;
    }

    private static long buildNegationStore(long inputNode, long resultNode) {
        assert (BitUtil.fits(inputNode, 25) && BitUtil.fits(resultNode, 25));
        return resultNode | inputNode << 32;
    }

    private static long getInputNodeFromTernarySecondStore(long ternarySecondStore) {
        return ternarySecondStore >>> 25;
    }

    private static long getResultNodeFromTernarySecondStore(long ternarySecondStore) {
        return ternarySecondStore & CACHE_VALUE_MASK;
    }

    private static long getResultNodeFromNegationStore(long negationStore) {
        return negationStore & CACHE_VALUE_MASK;
    }

    private static long getNegationFullKeyFromNegationStore(long negationStore) {
        return negationStore & (BitUtil.maskLength(32) ^ 0xFFFFFFFFFFFFFFFFL);
    }

    private static void insertInLru(long[] array, int first, int offset, long newValue) {
        System.arraycopy(array, first, array, first + 1, offset - 1);
        array[first] = newValue;
    }

    private static void insertInLru(int[] array, int first, int offset, int newValue) {
        System.arraycopy(array, first, array, first + 1, offset - 1);
        array[first] = newValue;
    }

    private static void insertInTernaryLru(long[] array, int first, int offset, long newFirstValue, long newSecondValue) {
        System.arraycopy(array, first, array, first + 2, offset - 2);
        array[first] = newFirstValue;
        array[first + 1] = newSecondValue;
    }

    private static boolean isBinaryOperation(int operationId) {
        return operationId == 0 || operationId == 4 || operationId == 3 || operationId == 1 || operationId == 2 || operationId == 5 || operationId == 6;
    }

    private static boolean isTernaryOperation(int operationId) {
        return operationId == 0;
    }

    private static void updateLru(long[] array, int first, int offset) {
        BddCache.insertInLru(array, first, offset, array[first + offset]);
    }

    private static void updateLru(int[] array, int first, int offset) {
        BddCache.insertInLru(array, first, offset, array[first + offset]);
    }

    private static void updateTernaryLru(long[] array, int first, int offset) {
        BddCache.insertInTernaryLru(array, first, offset, array[first + offset], array[first + offset + 1]);
    }

    private int binaryHash(long binaryKey) {
        int binaryHashSize = this.getBinaryCacheKeyCount();
        int hash = MathUtil.hash(binaryKey) % binaryHashSize;
        return hash < 0 ? hash + binaryHashSize : hash;
    }

    private boolean binaryLookup(int operationId, int inputNode1, int inputNode2) {
        assert (this.associatedBdd.isNodeValid(inputNode1) && this.associatedBdd.isNodeValid(inputNode2));
        assert (BddCache.isBinaryOperation(operationId));
        long binaryKey = BddCache.buildBinaryKeyStore(operationId, inputNode1, inputNode2);
        this.lookupHash = this.binaryHash(binaryKey);
        int cachePosition = this.getBinaryCachePosition(this.lookupHash);
        if (this.binaryBinsPerHash == 1) {
            if (binaryKey != this.binaryKeyStorage[cachePosition]) {
                return false;
            }
            this.lookupResult = this.binaryResultStorage[cachePosition];
        } else {
            int offset = -1;
            for (int i = 0; i < this.binaryBinsPerHash; ++i) {
                long binaryKeyStore = this.binaryKeyStorage[cachePosition + i];
                if (binaryKey != binaryKeyStore) continue;
                offset = i;
                break;
            }
            if (offset == -1) {
                return false;
            }
            this.lookupResult = this.binaryResultStorage[cachePosition + offset];
            if (offset != 0) {
                BddCache.updateLru(this.binaryKeyStorage, cachePosition, offset);
                BddCache.updateLru(this.binaryResultStorage, cachePosition, offset);
            }
        }
        assert (this.associatedBdd.isNodeValidOrRoot(this.lookupResult));
        this.binaryAccessStatistics.cacheHit();
        return true;
    }

    private void binaryPut(int operationId, int hash, int inputNode1, int inputNode2, int resultNode) {
        assert (BddCache.isBinaryOperation(operationId) && this.associatedBdd.isNodeValid(inputNode1) && this.associatedBdd.isNodeValid(inputNode2) && this.associatedBdd.isNodeValidOrRoot(resultNode));
        int cachePosition = this.getBinaryCachePosition(hash);
        this.binaryAccessStatistics.put();
        long binaryKeyStore = BddCache.buildBinaryKeyStore(operationId, inputNode1, inputNode2);
        if (this.binaryBinsPerHash == 1) {
            this.binaryKeyStorage[cachePosition] = binaryKeyStore;
            this.binaryResultStorage[cachePosition] = resultNode;
        } else {
            BddCache.insertInLru(this.binaryKeyStorage, cachePosition, this.binaryBinsPerHash, binaryKeyStore);
            BddCache.insertInLru(this.binaryResultStorage, cachePosition, this.binaryBinsPerHash, resultNode);
        }
    }

    void clearVolatileCache() {
        this.volatileAccessStatistics.invalidation();
        Arrays.fill(this.volatileKeyStorage, 0);
    }

    private int composeHash(int inputNode, int[] replacementArray) {
        int composeHashSize = this.getComposeKeyCount();
        int hash = MathUtil.hash(inputNode, replacementArray) % composeHashSize;
        return hash < 0 ? hash + composeHashSize : hash;
    }

    private float computeBinaryLoadFactor() {
        int loadedBinaryBins = 0;
        for (int i = 0; i < this.getBinaryCacheKeyCount(); ++i) {
            if (this.binaryKeyStorage[i] == 0L) continue;
            ++loadedBinaryBins;
        }
        return (float)loadedBinaryBins / (float)this.getBinaryCacheKeyCount();
    }

    private float computeSatisfactionLoadFactor() {
        int loadedSatisfactionBins = 0;
        for (int i = 0; i < this.getSatisfactionKeyCount(); ++i) {
            if ((long)this.satisfactionKeyStorage[i] == 0L) continue;
            ++loadedSatisfactionBins;
        }
        return (float)loadedSatisfactionBins / (float)this.getSatisfactionKeyCount();
    }

    private float computeTernaryLoadFactor() {
        int loadedTernaryBins = 0;
        for (int i = 0; i < this.getTernaryKeyCount(); ++i) {
            if (this.ternaryStorage[i * 2] == 0L) continue;
            ++loadedTernaryBins;
        }
        return (float)loadedTernaryBins / (float)this.getTernaryKeyCount();
    }

    private float computeNegationLoadFactor() {
        int loadedNegationBins = 0;
        for (int i = 0; i < this.getNegationCacheKeyCount(); ++i) {
            if (this.negationStorage[i] == 0L) continue;
            ++loadedNegationBins;
        }
        return (float)loadedNegationBins / (float)this.getNegationCacheKeyCount();
    }

    private float computeVolatileLoadFactor() {
        int loadedVolatileBins = 0;
        for (int i = 0; i < this.getVolatileKeyCount(); ++i) {
            if (this.volatileKeyStorage[i] == 0) continue;
            ++loadedVolatileBins;
        }
        return (float)loadedVolatileBins / (float)this.getVolatileKeyCount();
    }

    private int ensureMinimumCacheKeyCount(int cacheSize) {
        if (cacheSize < this.associatedBdd.getConfiguration().minimumNodeTableSize()) {
            return MathUtil.nextPrime(this.associatedBdd.getConfiguration().minimumNodeTableSize());
        }
        return MathUtil.nextPrime(cacheSize);
    }

    private int getBinaryCacheKeyCount() {
        return this.binaryKeyStorage.length / this.binaryBinsPerHash;
    }

    private int getBinaryCachePosition(int hash) {
        return hash * this.binaryBinsPerHash;
    }

    private int getComposeCachePosition(int hash) {
        return hash * (2 + this.associatedBdd.numberOfVariables() + this.composeBinsPerHash);
    }

    private int getComposeKeyCount() {
        assert (this.composeStorage != null);
        return this.composeStorage.length / (2 + this.composeBinsPerHash + this.associatedBdd.numberOfVariables());
    }

    int getLookupHash() {
        return this.lookupHash;
    }

    int getLookupResult() {
        return this.lookupResult;
    }

    private int getSatisfactionCachePosition(int hash) {
        return hash * this.satisfactionBinsPerHash;
    }

    private int getSatisfactionKeyCount() {
        return this.satisfactionKeyStorage.length / this.satisfactionBinsPerHash;
    }

    String getStatistics() {
        StringBuilder builder = new StringBuilder(512);
        builder.append("Negation: size: ").append(this.getNegationCacheKeyCount()).append(", load: ").append(this.computeNegationLoadFactor()).append("\n ").append(this.negationAccessStatistics).append("\nBinary: size: ").append(this.getBinaryCacheKeyCount()).append(", load: ").append(this.computeBinaryLoadFactor()).append("\n ").append(this.binaryAccessStatistics).append("\nTernary: size: ").append(this.getTernaryKeyCount()).append(", load: ").append(this.computeTernaryLoadFactor()).append("\n ").append(this.ternaryAccessStatistics).append("\nSatisfaction: size: ").append(this.getSatisfactionKeyCount()).append(", load: ").append(this.computeSatisfactionLoadFactor()).append("\n ").append(this.satisfactionAccessStatistics).append("\nCompose:");
        if (this.composeStorage == null) {
            builder.append(" Disabled");
        } else {
            builder.append(" size: ").append(this.getComposeKeyCount()).append("\n ").append(this.composeAccessStatistics);
        }
        builder.append("\nCompose volatile: current size: ").append(this.getVolatileKeyCount()).append(", load: ").append(this.computeVolatileLoadFactor()).append("\n ").append(this.volatileAccessStatistics);
        return builder.toString();
    }

    private int getTernaryCachePosition(int hash) {
        return hash * this.ternaryBinsPerHash * 2;
    }

    private int getTernaryKeyCount() {
        return this.ternaryStorage.length / this.ternaryBinsPerHash / 2;
    }

    private int getNegationCacheKeyCount() {
        return this.negationStorage.length / this.negationBinsPerHash;
    }

    private int getNegationCachePosition(int hash) {
        return hash * this.negationBinsPerHash;
    }

    private int getVolatileCachePosition(int hash) {
        return hash * this.volatileBinsPerHash;
    }

    private int getVolatileKeyCount() {
        return this.volatileKeyStorage.length / this.volatileBinsPerHash;
    }

    private int hashSatisfaction(int node) {
        int satisfactionHashSize = this.getSatisfactionKeyCount();
        int hash = MathUtil.hash(node) % satisfactionHashSize;
        return hash < 0 ? hash + satisfactionHashSize : hash;
    }

    void invalidate() {
        this.invalidateNegation();
        this.invalidateBinary();
        this.invalidateTernary();
        this.invalidateSatisfaction();
        this.invalidateCompose();
        this.clearVolatileCache();
    }

    void invalidateBinary() {
        this.binaryAccessStatistics.invalidation();
        this.reallocateBinary();
    }

    void invalidateCompose() {
        this.composeAccessStatistics.invalidation();
        this.reallocateCompose();
    }

    void invalidateSatisfaction() {
        this.satisfactionAccessStatistics.invalidation();
        this.reallocateSatisfaction();
    }

    void invalidateTernary() {
        this.ternaryAccessStatistics.invalidation();
        this.reallocateTernary();
    }

    void invalidateNegation() {
        this.negationAccessStatistics.invalidation();
        this.reallocateNegation();
    }

    boolean lookupAnd(int inputNode1, int inputNode2) {
        return this.binaryLookup(0, inputNode1, inputNode2);
    }

    boolean lookupCompose(int inputNode, int[] replacementArray) {
        assert (this.composeStorage != null);
        assert (this.associatedBdd.isNodeValid(inputNode));
        int hash = this.composeHash(inputNode, replacementArray);
        int cachePosition = this.getComposeCachePosition(hash);
        this.lookupHash = hash;
        if (this.composeStorage[cachePosition] != inputNode) {
            return false;
        }
        for (int i = 0; i < replacementArray.length; ++i) {
            if (this.composeStorage[cachePosition + 2 + i] == replacementArray[i]) continue;
            return false;
        }
        if (replacementArray.length < this.associatedBdd.numberOfVariables() && this.composeStorage[cachePosition + 2 + replacementArray.length] != -1) {
            return false;
        }
        this.lookupResult = this.composeStorage[cachePosition + 1];
        assert (this.associatedBdd.isNodeValidOrRoot(this.lookupResult));
        this.composeAccessStatistics.cacheHit();
        return true;
    }

    boolean lookupEquivalence(int inputNode1, int inputNode2) {
        return this.binaryLookup(4, inputNode1, inputNode2);
    }

    boolean lookupExists(int inputNode, int variableCube) {
        return this.binaryLookup(6, inputNode, variableCube);
    }

    boolean lookupIfThenElse(int inputNode1, int inputNode2, int inputNode3) {
        return this.ternaryLookup(0, inputNode1, inputNode2, inputNode3);
    }

    boolean lookupImplication(int inputNode1, int inputNode2) {
        return this.binaryLookup(3, inputNode1, inputNode2);
    }

    boolean lookupNAnd(int inputNode1, int inputNode2) {
        return this.binaryLookup(1, inputNode1, inputNode2);
    }

    boolean lookupNot(int node) {
        return this.negationLookup(node);
    }

    boolean lookupOr(int inputNode1, int inputNode2) {
        return this.binaryLookup(2, inputNode1, inputNode2);
    }

    double lookupSatisfaction(int node) {
        assert (this.associatedBdd.isNodeValid(node));
        int hash = this.hashSatisfaction(node);
        int cachePosition = this.getSatisfactionCachePosition(hash);
        this.lookupHash = hash;
        if (node == this.satisfactionKeyStorage[cachePosition]) {
            this.satisfactionAccessStatistics.cacheHit();
            return this.satisfactionResultStorage[cachePosition];
        }
        return -1.0;
    }

    boolean lookupVolatile(int inputNode) {
        assert (this.associatedBdd.isNodeValid(inputNode));
        this.lookupHash = this.volatileHash(inputNode);
        int cachePosition = this.getVolatileCachePosition(this.lookupHash);
        if (this.volatileBinsPerHash == 1) {
            if (this.volatileKeyStorage[cachePosition] != inputNode) {
                return false;
            }
            this.lookupResult = this.volatileResultStorage[cachePosition];
        } else {
            int offset = -1;
            for (int i = 0; i < this.volatileBinsPerHash; ++i) {
                int keyValue = this.volatileKeyStorage[cachePosition + i];
                if (keyValue == 0) {
                    return false;
                }
                if (keyValue != inputNode) continue;
                offset = i;
                break;
            }
            if (offset == -1) {
                return false;
            }
            this.lookupResult = this.volatileResultStorage[cachePosition + offset];
            if (offset != 0) {
                BddCache.updateLru(this.volatileKeyStorage, cachePosition, offset);
                BddCache.updateLru(this.volatileResultStorage, cachePosition, offset);
            }
        }
        assert (this.associatedBdd.isNodeValidOrRoot(this.lookupResult));
        this.volatileAccessStatistics.cacheHit();
        return true;
    }

    boolean lookupXor(int inputNode1, int inputNode2) {
        return this.binaryLookup(5, inputNode1, inputNode2);
    }

    void putAnd(int hash, int inputNode1, int inputNode2, int resultNode) {
        this.binaryPut(0, hash, inputNode1, inputNode2, resultNode);
    }

    void putCompose(int hash, int inputNode, int[] replacement, int resultNode) {
        assert (this.composeStorage != null);
        assert (this.associatedBdd.isNodeValidOrRoot(inputNode) && this.associatedBdd.isNodeValidOrRoot(resultNode));
        assert (replacement.length <= this.associatedBdd.numberOfVariables());
        int cachePosition = this.getComposeCachePosition(hash);
        this.composeAccessStatistics.put();
        this.composeStorage[cachePosition] = inputNode;
        this.composeStorage[cachePosition + 1] = resultNode;
        System.arraycopy(replacement, 0, this.composeStorage, cachePosition + 2, replacement.length);
        if (replacement.length < this.associatedBdd.numberOfVariables()) {
            this.composeStorage[cachePosition + 2 + replacement.length] = -1;
        }
    }

    void putEquivalence(int hash, int inputNode1, int inputNode2, int resultNode) {
        this.binaryPut(4, hash, inputNode1, inputNode2, resultNode);
    }

    void putExists(int hash, int inputNode, int variableCube, int resultNode) {
        this.binaryPut(6, hash, inputNode, variableCube, resultNode);
    }

    void putIfThenElse(int hash, int inputNode1, int inputNode2, int inputNode3, int resultNode) {
        this.ternaryPut(0, hash, inputNode1, inputNode2, inputNode3, resultNode);
    }

    void putImplication(int hash, int inputNode1, int inputNode2, int resultNode) {
        this.binaryPut(3, hash, inputNode1, inputNode2, resultNode);
    }

    void putNAnd(int hash, int inputNode1, int inputNode2, int resultNode) {
        this.binaryPut(1, hash, inputNode1, inputNode2, resultNode);
    }

    void putNot(int inputNode, int resultNode) {
        this.negationPut(this.negationHash(BddCache.buildNegationFullKey(inputNode)), inputNode, resultNode);
    }

    void putNot(int hash, int inputNode, int resultNode) {
        assert (this.associatedBdd.isNodeValid(inputNode) && this.associatedBdd.isNodeValidOrRoot(resultNode));
        this.negationPut(hash, inputNode, resultNode);
    }

    void putOr(int hash, int inputNode1, int inputNode2, int resultNode) {
        this.binaryPut(2, hash, inputNode1, inputNode2, resultNode);
    }

    void putSatisfaction(int hash, int node, double satisfactionCount) {
        assert (this.associatedBdd.isNodeValid(node));
        int cachePosition = this.getSatisfactionCachePosition(hash);
        this.satisfactionKeyStorage[cachePosition] = node;
        this.satisfactionResultStorage[cachePosition] = satisfactionCount;
        this.satisfactionAccessStatistics.put();
    }

    void putVolatile(int hash, int inputNode, int resultNode) {
        assert (this.associatedBdd.isNodeValid(inputNode) && this.associatedBdd.isNodeValidOrRoot(resultNode));
        int cachePosition = this.getVolatileCachePosition(hash);
        this.volatileAccessStatistics.put();
        if (this.volatileBinsPerHash == 1) {
            this.volatileKeyStorage[cachePosition] = inputNode;
            this.volatileResultStorage[cachePosition] = resultNode;
        } else {
            BddCache.insertInLru(this.volatileKeyStorage, cachePosition, this.volatileBinsPerHash, inputNode);
            BddCache.insertInLru(this.volatileResultStorage, cachePosition, this.volatileBinsPerHash, resultNode);
        }
    }

    void putXor(int hash, int inputNode1, int inputNode2, int resultNode) {
        this.binaryPut(5, hash, inputNode1, inputNode2, resultNode);
    }

    private void reallocateBinary() {
        int keyCount = this.associatedBdd.getTableSize() / this.associatedBdd.getConfiguration().cacheBinaryDivider();
        int actualSize = this.ensureMinimumCacheKeyCount(keyCount) * this.binaryBinsPerHash;
        this.binaryKeyStorage = new long[actualSize];
        this.binaryResultStorage = new int[actualSize];
    }

    private void reallocateCompose() {
        if (!this.associatedBdd.getConfiguration().useGlobalComposeCache()) {
            this.composeStorage = null;
            return;
        }
        int keyCount = this.associatedBdd.getTableSize() / this.associatedBdd.getConfiguration().cacheComposeDivider();
        int actualSize = this.ensureMinimumCacheKeyCount(keyCount) * (2 + this.associatedBdd.numberOfVariables());
        this.composeStorage = new int[actualSize];
    }

    private void reallocateSatisfaction() {
        int keyCount = this.associatedBdd.getTableSize() / this.associatedBdd.getConfiguration().cacheSatisfactionDivider();
        int actualSize = this.ensureMinimumCacheKeyCount(keyCount) * this.satisfactionBinsPerHash;
        this.satisfactionKeyStorage = new int[actualSize];
        this.satisfactionResultStorage = new double[actualSize];
    }

    private void reallocateTernary() {
        int keyCount = this.associatedBdd.getTableSize() / this.associatedBdd.getConfiguration().cacheTernaryDivider();
        int actualSize = this.ensureMinimumCacheKeyCount(keyCount) * this.ternaryBinsPerHash * 2;
        this.ternaryStorage = new long[actualSize];
    }

    private void reallocateNegation() {
        int keyCount = this.associatedBdd.getTableSize() / this.associatedBdd.getConfiguration().cacheNegationDivider();
        int actualSize = this.ensureMinimumCacheKeyCount(keyCount) * this.negationBinsPerHash;
        this.negationStorage = new long[actualSize];
    }

    void reallocateVolatile() {
        int keyCount = this.associatedBdd.numberOfVariables() * this.associatedBdd.getConfiguration().cacheVolatileMultiplier();
        this.volatileAccessStatistics.invalidation();
        int actualSize = MathUtil.nextPrime(keyCount) * this.volatileBinsPerHash;
        this.volatileKeyStorage = new int[actualSize];
        this.volatileResultStorage = new int[actualSize];
    }

    private int ternaryHash(long ternaryFirstStore, int inputNode3) {
        int ternaryHashSize = this.getTernaryKeyCount();
        int hash = MathUtil.hash(ternaryFirstStore, inputNode3) % ternaryHashSize;
        return hash < 0 ? hash + ternaryHashSize : hash;
    }

    private boolean ternaryLookup(int operationId, int inputNode1, int inputNode2, int inputNode3) {
        assert (BddCache.isTernaryOperation(operationId) && this.associatedBdd.isNodeValid(inputNode1) && this.associatedBdd.isNodeValid(inputNode2) && this.associatedBdd.isNodeValid(inputNode3));
        assert (BddCache.isTernaryOperation(operationId));
        long constructedTernaryFirstStore = BddCache.buildTernaryFirstStore(operationId, inputNode1, inputNode2);
        this.lookupHash = this.ternaryHash(constructedTernaryFirstStore, inputNode3);
        int cachePosition = this.getTernaryCachePosition(this.lookupHash);
        if (this.ternaryBinsPerHash == 1) {
            if (constructedTernaryFirstStore != this.ternaryStorage[cachePosition]) {
                return false;
            }
            long ternarySecondStore = this.ternaryStorage[cachePosition + 1];
            if (inputNode3 != (int)BddCache.getInputNodeFromTernarySecondStore(ternarySecondStore)) {
                return false;
            }
            this.lookupResult = (int)BddCache.getResultNodeFromTernarySecondStore(ternarySecondStore);
        } else {
            int offset = -1;
            for (int i = 0; i < this.ternaryBinsPerHash * 2; i += 2) {
                long ternarySecondStore;
                if (constructedTernaryFirstStore != this.ternaryStorage[cachePosition + i] || inputNode3 != (int)BddCache.getInputNodeFromTernarySecondStore(ternarySecondStore = this.ternaryStorage[cachePosition + i + 1])) continue;
                offset = i;
                this.lookupResult = (int)BddCache.getResultNodeFromTernarySecondStore(ternarySecondStore);
                break;
            }
            if (offset == -1) {
                return false;
            }
            if (offset != 0) {
                BddCache.updateTernaryLru(this.ternaryStorage, cachePosition, offset);
            }
        }
        assert (this.associatedBdd.isNodeValidOrRoot(this.lookupResult));
        this.ternaryAccessStatistics.cacheHit();
        return true;
    }

    private void ternaryPut(int operationId, int hash, int inputNode1, int inputNode2, int inputNode3, int resultNode) {
        assert (this.associatedBdd.isNodeValid(inputNode1) && this.associatedBdd.isNodeValid(inputNode2) && this.associatedBdd.isNodeValid(inputNode3) && this.associatedBdd.isNodeValidOrRoot(resultNode));
        assert (BddCache.isTernaryOperation(operationId));
        int cachePosition = this.getTernaryCachePosition(hash);
        this.ternaryAccessStatistics.put();
        long firstStore = BddCache.buildTernaryFirstStore(operationId, inputNode1, inputNode2);
        long secondStore = BddCache.buildTernarySecondStore(inputNode3, resultNode);
        if (this.ternaryBinsPerHash == 1) {
            this.ternaryStorage[cachePosition] = firstStore;
            this.ternaryStorage[cachePosition + 1] = secondStore;
        } else {
            BddCache.insertInTernaryLru(this.ternaryStorage, cachePosition, this.ternaryBinsPerHash * 2, firstStore, secondStore);
        }
    }

    private int negationHash(long negationKey) {
        int negationHashSize = this.getNegationCacheKeyCount();
        int hash = MathUtil.hash(negationKey) % negationHashSize;
        return hash < 0 ? hash + negationHashSize : hash;
    }

    private boolean negationLookup(int inputNode) {
        assert (this.associatedBdd.isNodeValid(inputNode));
        long negationFullKey = BddCache.buildNegationFullKey(inputNode);
        this.lookupHash = this.negationHash(negationFullKey);
        int cachePosition = this.getNegationCachePosition(this.lookupHash);
        if (this.negationBinsPerHash == 1) {
            long negationStore = this.negationStorage[cachePosition];
            long negationStoreFullKey = BddCache.getNegationFullKeyFromNegationStore(negationStore);
            if (negationFullKey != negationStoreFullKey) {
                return false;
            }
            this.lookupResult = (int)BddCache.getResultNodeFromNegationStore(negationStore);
        } else {
            int offset = -1;
            for (int i = 0; i < this.negationBinsPerHash; ++i) {
                long negationStore = this.negationStorage[cachePosition + i];
                long negationStoreFullKey = BddCache.getNegationFullKeyFromNegationStore(negationStore);
                if (negationFullKey != negationStoreFullKey) continue;
                offset = i;
                this.lookupResult = (int)BddCache.getResultNodeFromNegationStore(negationStore);
                break;
            }
            if (offset == -1) {
                return false;
            }
            if (offset != 0) {
                BddCache.updateLru(this.negationStorage, cachePosition, offset);
            }
        }
        assert (this.associatedBdd.isNodeValidOrRoot(this.lookupResult));
        this.negationAccessStatistics.cacheHit();
        return true;
    }

    private void negationPut(int hash, int inputNode, int resultNode) {
        assert (this.associatedBdd.isNodeValid(inputNode) && this.associatedBdd.isNodeValidOrRoot(resultNode));
        int cachePosition = this.getNegationCachePosition(hash);
        this.negationAccessStatistics.put();
        long negationStore = BddCache.buildNegationStore(inputNode, resultNode);
        if (this.negationBinsPerHash == 1) {
            this.negationStorage[cachePosition] = negationStore;
        } else {
            BddCache.insertInLru(this.negationStorage, cachePosition, this.negationBinsPerHash, negationStore);
        }
    }

    private int volatileHash(int inputNode) {
        int volatileHashSize = this.getVolatileKeyCount();
        int hash = MathUtil.hash(inputNode) % volatileHashSize;
        return hash < 0 ? hash + volatileHashSize : hash;
    }

    private static final class ShutdownHookPrinter
    implements Runnable {
        private ShutdownHookPrinter() {
        }

        @Override
        public void run() {
            if (!logger.isLoggable(Level.FINE)) {
                return;
            }
            for (BddCache cache : cacheShutdownHook) {
                logger.log(Level.FINE, cache.getStatistics());
            }
        }
    }

    private static final class ShutdownHookLazyHolder {
        private static final Runnable shutdownHook = new ShutdownHookPrinter();

        private ShutdownHookLazyHolder() {
        }

        static void init() {
        }

        static {
            Runtime.getRuntime().addShutdownHook(new Thread(shutdownHook));
        }
    }

    private static final class CacheAccessStatistics {
        private int hitCount = 0;
        private int hitCountSinceInvalidation = 0;
        private int invalidationCount = 0;
        private int putCount = 0;
        private int putCountSinceInvalidation = 0;

        private CacheAccessStatistics() {
        }

        void cacheHit() {
            ++this.hitCount;
            ++this.hitCountSinceInvalidation;
        }

        void invalidation() {
            ++this.invalidationCount;
            this.hitCountSinceInvalidation = 0;
            this.putCountSinceInvalidation = 0;
        }

        void put() {
            ++this.putCount;
            ++this.putCountSinceInvalidation;
        }

        public String toString() {
            float hitToPutRatio = (float)this.hitCount / (float)Math.max(this.putCount, 1);
            return String.format("Cache access: put=%d, hit=%d, hit-to-put=%3.3f%n       invalidation: %d times, since last: put=%d, hit=%d", this.putCount, this.hitCount, Float.valueOf(hitToPutRatio), this.invalidationCount, this.putCountSinceInvalidation, this.hitCountSinceInvalidation);
        }
    }
}

